# app.py

import os
import io
import numpy as np
import tensorflow as tf
from flask import Flask, request, jsonify, render_template_string
from PIL import Image

# Import the pre-trained VGG16 model.
# This model is loaded once when the application starts.
try:
    print("Loading VGG16 model...")
    vgg16_model = tf.keras.applications.VGG16(
        weights="imagenet",
        include_top=False,
        input_shape=(224, 224, 3)
    )
    print("VGG16 model loaded successfully.")
except Exception as e:
    print(f"Error loading VGG16 model: {e}")
    vgg16_model = None  # Handle cases where the model fails to load

app = Flask(__name__)

# A simple HTML template for the user interface
HTML_TEMPLATE = """
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Similar Image Search</title>
    <style>
        body { font-family: sans-serif; text-align: center; padding: 20px; }
        .container { max-width: 600px; margin: auto; padding: 20px; border-radius: 10px; box-shadow: 0 0 10px rgba(0,0,0,0.1); }
        h1 { color: #333; }
        input[type="file"] { margin: 10px 0; }
        button { padding: 10px 20px; background-color: #007BFF; color: white; border: none; border-radius: 5px; cursor: pointer; }
        button:hover { background-color: #0056b3; }
        #results { margin-top: 20px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>Similar Image Search</h1>
        <p>Upload an image to find similar ones.</p>
        <form id="upload-form" enctype="multipart/form-data">
            <input type="file" name="image" id="image-input" accept="image/*" required>
            <button type="submit">Search</button>
        </form>
        <div id="results"></div>
    </div>
    <script>
        const form = document.getElementById('upload-form');
        const resultsDiv = document.getElementById('results');

        form.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            resultsDiv.innerHTML = '<p>Processing... This might take a moment.</p>';
            
            const formData = new FormData(form);
            
            try {
                const response = await fetch('/similar_image_search', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (response.ok) {
                    resultsDiv.innerHTML = '<p>Search successful!</p>';
                    // You would display the similar images here
                    console.log('Similar images:', data.results);
                } else {
                    resultsDiv.innerHTML = `<p style="color:red;">Error: ${data.error}</p>`;
                }
            } catch (error) {
                resultsDiv.innerHTML = `<p style="color:red;">An unexpected error occurred: ${error.message}</p>`;
                console.error('Fetch error:', error);
            }
        });
    </script>
</body>
</html>
"""

# A simple function to process the image and get features
def get_image_features(image_data):
    """
    Processes a raw image file into a feature vector.
    """
    if vgg16_model is None:
        return None
    
    try:
        image = Image.open(io.BytesIO(image_data)).convert("RGB").resize((224, 224))
        image = np.array(image)
        image = np.expand_dims(image, axis=0)
        image = tf.keras.applications.vgg16.preprocess_input(image)
        
        features = vgg16_model.predict(image)
        return features.flatten()
    except Exception as e:
        print(f"Error processing image: {e}")
        return None

@app.route("/", methods=["GET"])
def index():
    """
    Serves the main application page.
    """
    return render_template_string(HTML_TEMPLATE)

@app.route("/similar_image_search", methods=["POST"])
def similar_image_search():
    """
    API endpoint for similar image search.
    This route reuses the pre-loaded VGG16 model.
    """
    if 'image' not in request.files:
        return jsonify({"error": "No image file provided."}), 400
    
    file = request.files['image']
    if file.filename == '':
        return jsonify({"error": "No selected file."}), 400
    
    if file and vgg16_model:
        image_data = file.read()
        
        # Get the feature vector using the pre-loaded model
        features = get_image_features(image_data)
        
        if features is not None:
            # In a real application, you would compare these features
            # to a database of pre-computed features for other images.
            # For this example, we just return a success message and the features.
            return jsonify({
                "message": "Image processed successfully!",
                "features_shape": str(features.shape),
                # Note: Returning the full feature vector in a real app is bad practice
                # This is just for demonstration purposes.
                "results": "Search logic would go here."
            })
        else:
            return jsonify({"error": "Failed to process image."}), 500
    
    return jsonify({"error": "An internal server error occurred."}), 500

if __name__ == "__main__":
    # In production, use Gunicorn instead of this
    app.run(debug=True, port=5000)

```python
# gunicorn_config.py

# A configuration file for Gunicorn.
# This file tells Gunicorn how to run your Flask application.

# The number of worker processes to spawn.
# A good rule of thumb is 2-4 workers per core.
workers = 4

# The address and port to bind to.
# This should match what your Nginx/front-end server is configured for.
bind = "0.0.0.0:5000"

# The timeout for a worker process.
# A value of 120 seconds should be sufficient for model loading.
timeout = 120

# The name of the module where your Flask application lives.
# Replace 'app' with the name of your Python file (e.g., 'myapp').
# The 'app' at the end is the name of your Flask instance (e.g., `app = Flask(__name__)`).
wsgi_app = "app:app"

# Log to stdout/stderr.
# This helps with debugging in production environments.
accesslog = "-"
errorlog = "-"
```text
# requirements.txt

# List of required Python packages for your application.
Flask==2.0.3
gunicorn==20.1.0
numpy==1.20.0
Pillow==9.2.0
tensorflow-cpu==2.10.0
